(function () {
  'use strict';

  /**
   * 常量定义文件
   */

  // URL模式
  const URL_PATTERNS = {
    TRAVEL: /bxdh=CCBX\d+/, // 差旅报销
    EXPENSE: /bxdh=FYBX\d+/, // 费用报销
  };

  // 报销类型
  const REIMBURSEMENT_TYPES = {
    TRAVEL: 'travel',
    EXPENSE: 'expense',
  };

  // 消息类型
  const MESSAGE_TYPES = {
    // Content Scripts -> Background
    PAGE_DETECTED: 'page_detected',
    DATA_EXTRACTED: 'data_extracted',
    EXTRACTION_ERROR: 'extraction_error',

    // Popup -> Background
    REQUEST_STATUS: 'request_status',
    START_EXTRACTION: 'start_extraction',
    GENERATE_PDF: 'generate_pdf',

    // Background -> Content Scripts
    START_EXTRACT: 'start_extract',
    EXTRACT_COMPLETE: 'extract_complete',

    // Background -> Popup
    STATUS_UPDATE: 'status_update',
    PDF_GENERATED: 'pdf_generated',
  };

  // 处理状态
  const PROCESS_STATUS = {
    IDLE: 'idle',
    DETECTING: 'detecting',
    EXTRACTING: 'extracting',
    VALIDATING: 'validating',
    GENERATING: 'generating',
    COMPLETED: 'completed',
    ERROR: 'error',
  };

  // DOM选择器配置 - 基于胜意费控云实际页面结构
  const SELECTORS = {
    // 基础信息 - 根据页面实际class选择
    company: [
      '.fygsgsmc-dom.text-dom .cversionText',  // 费用归属公司
      '.company-name',
      '.el-page-header__content',
    ],
    // 员工信息
    employeeName: [
      '.bxrxm-dom.text-dom .cversionText',  // 报销人
    ],
    employeeId: [
      // 工号可能在span中显示为 "盛万春/DT0012"
      '.bxrxm-dom.text-dom .cversionText',
    ],
    department: [
      '.bxbmmc-dom.text-dom .cversionText',  // 报销部门
      '.fygsbmmc-dom.text-dom .cversionText',  // 费用归属部门
    ],
    departmentCode: [
      // 部门编码可能需要从其他元素获取
    ],

    // 金额信息
    actualAmount: [
      '.key-yfjje .right',  // 应付金额
      '.key-grdfje .right',  // 个人垫付金额
      '.fylb-amount',  // 费用明细合计
      '.bxje .right',  // 报销总金额
    ],
    totalAmount: [
      '.key-xfzje .right',  // 消费总金额
    ],
    voucherCount: [
      '.invoice-nums i',  // 票据数量
      '.cost-total-right i',  // 票据数量
    ]};

  // 性能配置
  const PERFORMANCE_CONFIG = {
    maxRetries: 3,
    retryDelay: 2000};

  /**
   * 日志工具类
   */

  class Logger {
    constructor(context = 'App') {
      this.context = context;
      this.logs = [];
      this.maxLogs = 1000;
    }

    log(level, message, data = {}) {
      const logEntry = {
        timestamp: new Date().toISOString(),
        level,
        context: this.context,
        message,
        data,
        url: typeof window !== 'undefined' ? window.location.href : '',
      };

      this.logs.push(logEntry);

      // 限制日志数量
      if (this.logs.length > this.maxLogs) {
        this.logs.shift();
      }

      // 控制台输出
      const logMethod = console[level] || console.log;
      logMethod(`[${this.context}] ${message}`, data);

      // 持久化到storage
      this.persist();
    }

    info(message, data) {
      this.log('info', message, data);
    }

    warn(message, data) {
      this.log('warn', message, data);
    }

    error(message, data) {
      this.log('error', message, data);
    }

    debug(message, data) {
      this.log('debug', message, data);
    }

    async persist() {
      if (typeof chrome !== 'undefined' && chrome.storage) {
        try {
          const key = `logs_${this.context}`;
          await chrome.storage.local.set({ [key]: this.logs.slice(-100) });
        } catch (error) {
          console.error('Failed to persist logs:', error);
        }
      }
    }

    async load() {
      if (typeof chrome !== 'undefined' && chrome.storage) {
        try {
          const key = `logs_${this.context}`;
          const result = await chrome.storage.local.get(key);
          if (result[key]) {
            this.logs = result[key];
          }
        } catch (error) {
          console.error('Failed to load logs:', error);
        }
      }
    }

    export() {
      return JSON.stringify(this.logs, null, 2);
    }

    clear() {
      this.logs = [];
      this.persist();
    }
  }

  /**
   * 页面检测器 - 检测当前页面是否为报销单页面
   */


  class PageDetector {
    constructor() {
      this.logger = new Logger('PageDetector');
    }

    /**
     * 检测页面类型
     * @returns {Object|null} {type: 'travel'|'expense', bxdh: string}
     */
    detectPage() {
      try {
        const url = window.location.href;
        const urlParams = new URLSearchParams(window.location.search);
        const bxdh = urlParams.get('bxdh');

        this.logger.info('开始检测页面', { url, bxdh });

        if (!bxdh) {
          this.logger.warn('URL中未找到bxdh参数');
          return null;
        }

        // 检测差旅报销
        if (URL_PATTERNS.TRAVEL.test(url)) {
          this.logger.info('检测到差旅报销页面', { bxdh });
          return {
            type: REIMBURSEMENT_TYPES.TRAVEL,
            bxdh: bxdh,
          };
        }

        // 检测费用报销
        if (URL_PATTERNS.EXPENSE.test(url)) {
          this.logger.info('检测到费用报销页面', { bxdh });
          return {
            type: REIMBURSEMENT_TYPES.EXPENSE,
            bxdh: bxdh,
          };
        }

        this.logger.warn('未识别的报销单类型', { bxdh });
        return null;
      } catch (error) {
        this.logger.error('页面检测失败', { error: error.message });
        return null;
      }
    }

    /**
     * 等待页面完全加载
     * @param {number} timeout 超时时间（毫秒）
     * @returns {Promise<boolean>}
     */
    async waitForPageReady(timeout = 15000) {
      this.logger.info('等待页面加载完成');

      return new Promise((resolve) => {
        // 如果已经加载完成
        if (document.readyState === 'complete') {
          this.logger.info('页面已加载完成');
          resolve(true);
          return;
        }

        let resolved = false;
        const timer = setTimeout(() => {
          if (!resolved) {
            resolved = true;
            this.logger.warn('页面加载超时');
            resolve(false);
          }
        }, timeout);

        // 监听加载完成事件
        const handleLoad = () => {
          if (!resolved) {
            resolved = true;
            clearTimeout(timer);
            this.logger.info('页面加载完成');
            resolve(true);
          }
        };

        window.addEventListener('load', handleLoad, { once: true });

        // 同时监听readystatechange
        document.addEventListener(
          'readystatechange',
          () => {
            if (document.readyState === 'complete') {
              handleLoad();
            }
          },
          { once: true }
        );
      });
    }

    /**
     * 等待报销单数据加载完成
     * 等待直到找到关键数据元素（如报销人信息、公司信息等）
     * @param {number} timeout 超时时间（毫秒）
     * @returns {Promise<boolean>}
     */
    async waitForDataReady(timeout = 15000) {
      this.logger.info('等待报销单数据加载');

      // 关键数据元素选择器（来自胜意费控云页面）
      const dataSelectors = [
        '.bxrxm-dom.text-dom .cversionText',  // 报销人
        '.fygsgsmc-dom.text-dom .cversionText',  // 费用归属公司
        '.bxbmmc-dom.text-dom .cversionText',  // 报销部门
        '.key-yfjje .right',  // 应付金额
        '.document-no',  // 单号信息
        '.expense-form',  // 报销表单
        '.reimbursement-form',  // 报销表单
      ];

      return new Promise((resolve) => {
        // 检查是否已经有数据
        for (const selector of dataSelectors) {
          const element = document.querySelector(selector);
          if (element && element.textContent.trim() && element.textContent.trim().length > 0) {
            const text = element.textContent.trim();
            // 检查是否有实际内容（不是占位符）
            if (text.length > 1 && !text.includes('未知') && !text.includes('加载中')) {
              this.logger.info('数据已就绪', { selector, text: text.substring(0, 20) });
              resolve(true);
              return;
            }
          }
        }

        // 观察DOM变化，等待数据加载
        let resolved = false;
        const observer = new MutationObserver((mutations) => {
          for (const selector of dataSelectors) {
            const element = document.querySelector(selector);
            if (element && element.textContent.trim()) {
              const text = element.textContent.trim();
              // 检查是否有实际内容
              if (text.length > 1 && !text.includes('未知') && !text.includes('加载中')) {
                if (!resolved) {
                  resolved = true;
                  observer.disconnect();
                  this.logger.info('数据加载完成', { selector, text: text.substring(0, 20) });
                  resolve(true);
                }
                return;
              }
            }
          }
        });

        observer.observe(document.body, {
          childList: true,
          subtree: true,
          characterData: true,
          characterDataOldValue: true,
        });

        // 超时处理
        const timer = setTimeout(() => {
          if (!resolved) {
            resolved = true;
            observer.disconnect();
            this.logger.warn('数据加载超时，继续执行');
            resolve(false);
          }
        }, timeout);

        // 清理函数
        const cleanup = () => {
          clearTimeout(timer);
          if (!resolved) {
            resolved = true;
            observer.disconnect();
          }
        };

        // 页面卸载时清理
        window.addEventListener('beforeunload', cleanup, { once: true });
      });
    }

    /**
     * 观察页面DOM变化
     * @param {Function} callback 回调函数
     * @returns {MutationObserver}
     */
    observePageChanges(callback) {
      this.logger.info('开始观察页面变化');

      const observer = new MutationObserver((mutations) => {
        this.logger.debug('检测到页面变化', { count: mutations.length });
        callback(mutations);
      });

      observer.observe(document.body, {
        childList: true,
        subtree: true,
        attributes: true,
        attributeFilter: ['class', 'style'],
      });

      return observer;
    }

    /**
     * 检查DOM特征
     * @returns {boolean}
     */
    checkDOMSignature() {
      const signatures = [
        '.expense-form',
        '.reimbursement-form',
        '#app',
        '.el-form',
      ];

      for (const selector of signatures) {
        if (document.querySelector(selector)) {
          this.logger.info('找到DOM特征', { selector });
          return true;
        }
      }

      this.logger.warn('未找到预期的DOM特征');
      return false;
    }
  }

  /**
   * 辅助工具函数
   */

  /**
   * 等待指定时间
   * @param {number} ms 毫秒数
   */
  function sleep(ms) {
    return new Promise((resolve) => setTimeout(resolve, ms));
  }

  /**
   * 清理文本（去除多余空格和特殊字符）
   * @param {string} text 原始文本
   */
  function sanitizeText(text) {
    if (typeof text !== 'string') return '';
    return text
      .replace(/<script[^>]*>.*?<\/script>/gi, '')
      .replace(/<iframe[^>]*>.*?<\/iframe>/gi, '')
      .replace(/on\w+="[^"]*"/gi, '')
      .replace(/\s+/g, ' ')
      .trim();
  }

  /**
   * 解析金额（支持多种格式）
   * @param {string|number} amount 金额
   */
  function parseAmount(amount) {
    if (typeof amount === 'number') return amount;
    if (typeof amount !== 'string') return 0;

    // 移除货币符号和逗号
    const cleaned = amount.replace(/[¥$,]/g, '').trim();
    const parsed = parseFloat(cleaned);

    return isNaN(parsed) ? 0 : Math.round(parsed * 100) / 100;
  }

  /**
   * 获取元素的文本内容
   * @param {Element} element DOM元素
   */
  function getElementText(element) {
    if (!element) return '';
    
    // 优先获取value（input元素）
    if (element.value) {
      return sanitizeText(element.value);
    }
    
    // 获取textContent
    return sanitizeText(element.textContent || element.innerText || '');
  }

  /**
   * 数据提取器 - 从页面DOM中提取报销数据
   * 支持从DOM和Vue/JS数据源提取
   */


  class DataExtractor {
    constructor() {
      this.logger = new Logger('DataExtractor');
      this.selectorCache = new Map();
      this.apiDataCache = null;
      this.networkInterceptor = null;
    }

    /**
     * 初始化网络拦截器
     * 拦截API响应以获取原始数据
     */
    initNetworkInterceptor() {
      if (this.networkInterceptor) return;

      const self = this;

      // 拦截XMLHttpRequest
      const originalXHROpen = XMLHttpRequest.prototype.open;
      const originalXHRSend = XMLHttpRequest.prototype.send;

      XMLHttpRequest.prototype.open = function(method, url) {
        this._apiUrl = url;
        return originalXHROpen.apply(this, arguments);
      };

      XMLHttpRequest.prototype.send = function(body) {
        this.addEventListener('load', function() {
          const url = this._apiUrl;

          // 只处理胜意费控云的请求
          if (!url || !url.includes('djrj.easyfees.cn')) return;

          // 检查是否是相关API
          if (!self.isRelevantApi(url)) return;

          // 检查状态码
          if (this.status !== 200) return;

          try {
            const response = JSON.parse(this.responseText);

            // 检查是否有result字段
            if (!response || !response.result) {
              self.logger.debug('API响应无result字段', { url });
              return;
            }

            const result = response.result;

            // 查找报销数据
            if (result.bxdzb && (result.bxmxList || result.skrlb)) {
              self.logger.info('✅ 从XHR拦截到报销数据', { url });
              self.apiDataCache = result;
            } else if (Array.isArray(result.bxmxList) && result.bxmxList.length > 0) {
              // 可能是明细列表
              self.logger.info('✅ 从XHR拦截到明细数据', { count: result.bxmxList.length, url });
              self.apiDataCache = result;
            } else {
              self.logger.debug('API响应结构', {
                url,
                hasBxdzb: !!result.bxdzb,
                hasBxmxList: Array.isArray(result.bxmxList),
                keys: Object.keys(result).slice(0, 10).join(',')
              });
            }
          } catch (e) {
            self.logger.debug('解析XHR响应失败', { url, error: e.message });
          }
        });
        return originalXHRSend.apply(this, arguments);
      };

      // 拦截fetch请求
      const originalFetch = window.fetch;
      window.fetch = async function(url, options) {
        const response = await originalFetch.apply(this, arguments);

        // 只处理胜意费控云的请求
        if (!url || !url.includes('djrj.easyfees.cn')) return response;

        // 检查是否是相关API
        if (!self.isRelevantApi(url)) return response;

        try {
          const clonedResponse = response.clone();
          const data = await clonedResponse.json();

          if (!data || !data.result) {
            return response;
          }

          const result = data.result;

          // 查找报销数据
          if (result.bxdzb && (result.bxmxList || result.skrlb)) {
            self.logger.info('✅ 从fetch拦截到报销数据', { url });
            self.apiDataCache = result;
          } else if (Array.isArray(result.bxmxList) && result.bxmxList.length > 0) {
            self.logger.info('✅ 从fetch拦截到明细数据', { count: result.bxmxList.length, url });
            self.apiDataCache = result;
          }
        } catch (e) {
          self.logger.debug('解析fetch响应失败', { url, error: e.message });
        }

        return response;
      };

      this.networkInterceptor = true;
      this.logger.info('网络拦截器已启动');

      // 尝试从Performance API获取已完成的请求
      this.checkExistingRequests();
    }

    /**
     * 检查已完成的请求并获取数据
     */
    checkExistingRequests() {
      try {
        const entries = performance.getEntriesByType('resource');
        for (const entry of entries) {
          if (entry.name && this.isRelevantApi(entry.name)) {
            this.logger.info('发现已完成的API请求', { url: entry.name });
            // 尝试获取响应数据
            this.fetchApiData(entry.name);
          }
        }
      } catch (e) {
        this.logger.debug('获取Performance API失败', { error: e.message });
      }
    }

    /**
     * 获取API数据
     */
    async fetchApiData(url) {
      try {
        this.logger.info('尝试获取API数据', { url });
        // 使用fetch尝试获取数据（忽略CORS错误）
        const response = await fetch(url, {
          method: 'GET',
          mode: 'cors',
          credentials: 'include'
        });
        if (response.ok) {
          const data = await response.json();
          if (data.result && (data.result.bxdzb || data.result.bxmxList)) {
            this.logger.info('✅ 从已完成请求获取到数据', { url });
            this.apiDataCache = data.result;
          }
        }
      } catch (e) {
        this.logger.debug('获取API数据失败', { url, error: e.message });
      }
    }

    /**
     * 判断是否为相关API
     */
    isRelevantApi(url) {
      if (!url) return false;
      // 检查是否是报销单相关API
      const isRelevant = (
        /COMM_B2G/i.test(url) ||      // 胜意费控云API前缀
        /loadBxd/i.test(url) ||       // 加载报销单
        /getBxd/i.test(url) ||        // 获取报销单
        /bxdzb/i.test(url) ||         // 报销单主表
        /bxmx/i.test(url) ||          // 报销明细
        /bxmxList/i.test(url) ||      // 报销明细列表
        /fybx/i.test(url)             // 费用报销
      );
      return isRelevant;
    }

    /**
     * 提取完整数据
     * @param {Object} pageInfo 页面信息
     * @returns {Promise<Object>}
     */
    async extract(pageInfo) {
      console.log('=== DataExtractor - 开始提取数据 ===');
      this.logger.info('开始提取数据', pageInfo);

      try {
        // 优先从Vue/JS数据源提取
        let jsData = await this.extractFromVueData();
        console.log('📊 Vue/JS数据提取结果:', jsData);

        // 🔍 提取费用类别（新增功能）
        const feeCategory = await this.extractFeeCategory(jsData);
        console.log('💰 费用类别提取结果:', feeCategory);

        const data = {
          basic: await this.extractBasicInfo(pageInfo, jsData),
          employee: await this.extractEmployeeInfo(jsData),
          details: await this.extractDetails(jsData),
          summary: await this.extractSummary(jsData),
          payment: await this.extractPayment(),
          meta: {
            extractTime: new Date().toISOString(),
            pageUrl: window.location.href,
            version: '1.0.0',
          },
        };

        console.log('📋 基础数据提取完成:', data);

        // 🆕 构建完整的bxdzb和bxmxList结构（兼容后端处理）
        const bxdzb = this.buildBxdzb(data, jsData, feeCategory);
        const bxmxList = this.buildBxmxList(data, jsData, feeCategory);

        // 🆕 将bxdzb和bxmxList添加到数据结构中
        data.bxdzb = bxdzb;
        data.bxmxList = bxmxList;

        console.log('🏗️ 构建的bxdzb:', bxdzb);
        console.log('🏗️ 构建的bxmxList:', bxmxList);

        // 输出最终提取结果汇总
        this.logger.info('🎯 最终提取结果汇总:');
        this.logger.info('完整提取数据:', data);
        
        // 从不同数据源汇总关键信息
        const bxdh = data.basic?.bxdh || jsData?.bxdzb?.bxdh || bxdzb?.bxdh || 'N/A';
        const bxrxm = data.employee?.name || jsData?.bxdzb?.bxrxm || bxdzb?.bxrxm || 'N/A';
        const bxrgh = data.employee?.empNo || jsData?.bxdzb?.bxrgh || bxdzb?.bxrgh || 'N/A';
        const company = data.basic?.company || jsData?.bxdzb?.fygsgsmc || bxdzb?.fygsgsmc || 'N/A';
        const department = data.employee?.department || jsData?.bxdzb?.bmbmmc || bxdzb?.bmbmmc || 'N/A';
        const amount = data.summary?.totalAmount || jsData?.bxdzb?.bxje || bxdzb?.bxje || 'N/A';
        const fpsl = jsData?.bxdzb?.fpsl || bxdzb?.fpsl || (data.details?.length || 'N/A');
        const detailCount = data.details?.length || bxmxList?.length || 0;
        const fylb = feeCategory || jsData?.bxdzb?.fylb || bxdzb?.fylb || 'N/A';
        
        console.log('📊 关键数据汇总:');
        console.log(`  - 报销单号: ${bxdh}`);
        console.log(`  - 报销人: ${bxrxm}`);
        console.log(`  - 工号: ${bxrgh}`);
        console.log(`  - 部门: ${department}`);
        console.log(`  - 公司: ${company}`);
        console.log(`  - 金额: ${amount}`);
        console.log(`  - 发票数量: ${fpsl}张`);
        console.log(`  - 费用类别: ${fylb}`);
        console.log(`  - 明细条数: ${detailCount}`);

        console.log('=== DataExtractor - 数据提取完成 ===');
        this.logger.info('数据提取完成', { data });
        return data;
      } catch (error) {
        console.log('❌ DataExtractor数据提取失败:', error.message);
        console.log('🔍 错误堆栈:', error.stack);
        this.logger.error('数据提取失败', { error: error.message });
        throw error;
      }
    }

    /**
     * 从Vue/JS数据源提取数据
     * 优先从网络拦截器或DOM获取数据
     */
    async extractFromVueData() {
      this.logger.info('尝试从Vue/JS数据源提取');

      // 1. 优先使用网络拦截器缓存的数据
      if (this.apiDataCache) {
        this.logger.info('✅ 使用网络拦截器缓存的数据');
        return this.apiDataCache;
      }

      // 2. 查找全局变量中的报销数据
      const globalData = this.findGlobalData();
      if (globalData) {
        this.logger.info('找到全局变量数据');
        return globalData;
      }

      // 3. 查找DOM中的数据
      const domData = this.findFromDOM();
      if (domData) {
        this.logger.info('找到DOM数据');
        return domData;
      }

      this.logger.warn('未找到Vue/JS数据源');
      return { bxdzb: null, bxmxList: [] };
    }

    /**
     * 从DOM中查找数据
     */
    findFromDOM() {
      // 查找包含JSON数据的script标签
      const scripts = document.querySelectorAll('script[type="application/json"], script[data-data]');
      for (const script of scripts) {
        try {
          const data = JSON.parse(script.textContent);
          if (data && (data.bxdzb || data.bxmxList)) {
            return data;
          }
        } catch (e) {
          // 忽略解析错误
        }
      }

      // 查找Vue组件中的data属性
      const vueElements = document.querySelectorAll('[class*="vue"], [data-vue]');
      for (const el of vueElements) {
        // 检查元素的__vue__属性
        if (el.__vue__) {
          const data = this.searchVueData(el.__vue__);
          if (data) return data;
        }
      }

      // 查找包含bxdzb或bxmxList的隐藏元素
      const hiddenElements = document.querySelectorAll('.hidden, [style*="display: none"], [type="hidden"]');
      for (const el of hiddenElements) {
        const text = el.textContent;
        if (text && (text.includes('"bxdzb"') || text.includes('"bxmxList"'))) {
          try {
            const data = JSON.parse(text);
            if (data && (data.bxdzb || data.bxmxList)) {
              return data;
            }
          } catch (e) {
            // 忽略解析错误
          }
        }
      }

      return null;
    }

    /**
     * 查找Vue组件实例数据
     */
    findVueData() {
      // 查找Vue 3应用
      const vue3App = document.querySelector('[data-v-app]');
      if (vue3App && window.__vue_app__) {
        try {
          const apps = window.__vue_app_;
          apps.forEach((app) => {
            const instance = app._instance;
            if (instance && instance.proxy) {
              const data = this.searchVueData(instance.proxy);
              if (data) return data;
            }
          });
        } catch (e) {
          this.logger.debug('Vue 3数据查找失败', e);
        }
      }

      // 查找Vue 2应用 - 检查所有元素的__vue__属性
      const allElements = document.querySelectorAll('*');
      for (const el of allElements) {
        if (el.__vue__) {
          const data = this.searchVueData(el.__vue__);
          if (data) return data;
        }
      }

      // 查找__vue__全局变量
      if (window.__vue__) {
        const data = this.searchVueData(window.__vue__);
        if (data) return data;
      }

      // 查找Vue.use或Vue实例
      if (window.Vue) {
        // Vue 2
        if (window.Vue.prototype && window.Vue.prototype.$data) {
          const data = this.searchVueData(window.Vue.prototype.$data);
          if (data) return data;
        }
      }

      // 查找所有可能包含Vue数据的全局变量
      for (const key of Object.keys(window)) {
        const obj = window[key];
        if (obj && typeof obj === 'object') {
          // 检查是否有$store、$data、data等属性
          if (obj.$store || obj.$data || obj.data || obj._isVue) {
            const data = this.searchVueData(obj);
            if (data) return data;
          }
        }
      }

      return null;
    }

    /**
     * 递归搜索Vue数据中的报销数据
     */
    searchVueData(obj, visited = new Set()) {
      if (!obj || typeof obj !== 'object') return null;
      if (visited.has(obj)) return null;
      visited.add(obj);

      // 检查当前对象是否包含报销数据特征
      if (this.isReimbursementData(obj)) {
        return obj;
      }

      // 递归搜索子对象
      for (const key of Object.keys(obj)) {
        try {
          const value = obj[key];
          if (value && typeof value === 'object') {
            const result = this.searchVueData(value, visited);
            if (result) return result;
          }
        } catch (e) {
          // 忽略访问错误
        }
      }

      return null;
    }

    /**
     * 判断是否为报销数据
     */
    isReimbursementData(obj) {
      if (!obj) return false;

      // 检查是否包含报销单特征字段
      const hasBxdzb = obj.bxdzb && typeof obj.bxdzb === 'object';
      const hasBxmxList = Array.isArray(obj.bxmxList);
      const hasSkrList = Array.isArray(obj.skrList);

      // 检查bxdzb中的特征字段
      if (hasBxdzb) {
        const bxdzb = obj.bxdzb;
        const hasBxrxm = typeof bxdzb.bxrxm === 'string';
        const hasBxje = typeof bxdzb.bxje === 'number' || typeof bxdzb.bxje === 'string';
        const hasBxdh = typeof bxdzb.bxdh === 'string';
        if ((hasBxrxm || hasBxdh) && hasBxje) return true;
      }

      // 检查是否有明细列表
      if (hasBxmxList && obj.bxmxList.length > 0) {
        const firstItem = obj.bxmxList[0];
        const hasFymc = typeof firstItem.fymc === 'string' || typeof firstItem.fymc === 'number';
        const hasFyje = typeof firstItem.fyje === 'number' || typeof firstItem.fyje === 'string';
        if (hasFymc && hasFyje) return true;
      }

      // 检查是否直接包含报销单信息（无嵌套bxdzb）
      if (obj.bxdh && obj.bxje) {
        return true;
      }

      // 检查差旅报销数据（可能有不同的字段名）
      if (obj.cclx || obj.cclxmc) {
        // 差旅类型字段存在，可能是差旅数据
        if (obj.djbh || obj.djje || obj.sjje) {
          return true;
        }
      }

      // 检查是否有费用明细且包含金额
      if (Array.isArray(obj.fymxList) && obj.fymxList.length > 0) {
        const first = obj.fymxList[0];
        if ((first.fymc || first.fylx) && (first.fyje || first.je)) {
          return true;
        }
      }

      return hasBxdzb && (hasBxmxList || hasSkrList);
    }

    /**
     * 深度扫描所有对象寻找数据（调试用）
     */
    deepScanForData() {
      this.logger.info('开始深度扫描...');
      const found = [];
      const visited = new Set();
      let scannedCount = 0;

      const scan = (obj, path) => {
        if (!obj || typeof obj !== 'object') return;
        if (visited.has(obj)) return;
        visited.add(obj);
        scannedCount++;

        // 检查是否是报销数据
        if (this.isReimbursementData(obj)) {
          const info = {
            path,
            hasBxdzb: !!obj.bxdzb,
            bxmxListCount: Array.isArray(obj.bxmxList) ? obj.bxmxList.length : 0,
            hasBxdh: !!obj.bxdh,
          };
          found.push(info);
          this.logger.info('✅ 找到报销数据', info);
        }

        // 递归扫描（限制深度）
        if (path.split('.').length < 8) {
          for (const key of Object.keys(obj)) {
            try {
              const value = obj[key];
              if (value && typeof value === 'object') {
                scan(value, path + '.' + key);
              }
            } catch (e) {
              // 忽略错误
            }
          }
        }
      };

      // 从常见的可能包含数据的对象开始扫描
      const rootObjects = [window];

      for (const root of rootObjects) {
        scan(root, 'window');
      }

      this.logger.info('扫描完成', { scannedObjects: scannedCount, foundCount: found.length });
      return found;
    }

    /**
     * 查找全局变量中的报销数据
     */
    findGlobalData() {
      this.logger.debug('开始搜索全局变量中的数据...');

      // 优先检查我们的修复脚本注入的数据
      const fixedData = window.__REIMBURSEMENT_DATA__ || window.reimbursementData || window.bxData || window.extractedData;
      if (fixedData && (fixedData.bxdzb || fixedData.bxmxList)) {
        this.logger.info('✅ 从修复脚本找到数据', { 
          hasBxdzb: !!fixedData.bxdzb,
          hasBxmxList: Array.isArray(fixedData.bxmxList),
          detailCount: Array.isArray(fixedData.bxmxList) ? fixedData.bxmxList.length : 0
        });
        return fixedData;
      }

      // 如果没有修复脚本的数据，直接从DOM提取
      this.logger.info('修复脚本数据不存在，直接从DOM提取数据');
      const domData = this.extractDataFromDOM();
      if (domData) {
        this.logger.info('✅ 从DOM提取到数据', {
          hasBxdzb: !!domData.bxdzb,
          hasBxmxList: Array.isArray(domData.bxmxList),
          detailCount: Array.isArray(domData.bxmxList) ? domData.bxmxList.length : 0
        });
        return domData;
      }

      // 查找常见的全局变量名
      const globalNames = [
        'reimbursementData',
        'bxData',
        'expenseData',
        'formData',
        'detailData',
        'pageData',
        'billData',
        'reimburseInfo',
        'costInfo',
        'bxdInfo',
        'loadBxdBaseInfo',
        'bxdzb',
        'bxmxList',
        'result',
        'app',
        'vue',
        'VueApp',
        'store',
        '$store',
      ];

      for (const name of globalNames) {
        if (window[name]) {
          this.logger.debug('找到全局变量', { name, type: typeof window[name] });
          const data = this.searchVueData(window[name]);
          if (data) {
            this.logger.info('✅ 从全局变量找到数据', { name });
            return data;
          }
        }
      }

      // 查找Vue实例
      const vueApp = document.querySelector('[data-v-app]');
      if (vueApp && vueApp.__vue_app__) {
        const app = vueApp.__vue_app__;
        this.logger.debug('找到Vue 3应用', { hasInstance: !!app._instance });
        if (app._instance) {
          const data = this.searchVueData(app._instance);
          if (data) {
            this.logger.info('✅ 从Vue 3实例找到数据');
            return data;
          }
        }
      }

      // 查找Vuex store
      if (window.__VUE_DEVTOOLS_GLOBAL_HOOK__) {
        this.logger.debug('找到Vue DevTools hook');
        const devtools = window.__VUE_DEVTOOLS_GLOBAL_HOOK__;
        if (devtools.apps) {
          for (const app of devtools.apps) {
            if (app._store) {
              this.logger.debug('找到Vuex store', { hasStore: !!app._store });
              const data = this.searchVueData(app._store);
              if (data) {
                this.logger.info('✅ 从Vuex store找到数据');
                return data;
              }
            }
          }
        }
      }

      // 查找所有Vue组件实例
      const allElements = document.querySelectorAll('*');
      let vueInstanceCount = 0;
      for (const el of allElements) {
        if (el.__vue__) {
          vueInstanceCount++;
          const data = this.searchVueData(el.__vue__);
          if (data) {
            this.logger.info('✅ 从Vue组件实例找到数据', { count: vueInstanceCount });
            return data;
          }
        }
      }
      this.logger.debug('找到Vue组件实例数量', { count: vueInstanceCount });

      // 查找所有包含数据的对象
      for (const key of Object.keys(window)) {
        const obj = window[key];
        if (obj && typeof obj === 'object' && !key.startsWith('_') && !key.startsWith('$')) {
          try {
            // 检查对象属性数量（可能包含数据的对象通常有较多属性）
            const keys = Object.keys(obj);
            if (keys.length > 20 && keys.some(k => k.includes('bx') || k.includes('Data') || k.includes('Info'))) {
              this.logger.debug('检查潜在数据对象', { key, keyCount: keys.length });
              if (this.isReimbursementData(obj)) {
                this.logger.info('✅ 从全局对象找到数据', { key });
                return obj;
              }
            }
          } catch (e) {
            // 忽略错误
          }
        }
      }

      this.logger.warn('未在全局变量中找到数据');
      return null;
    }

    /**
     * 直接从DOM提取报销数据
     */
    extractDataFromDOM() {
      this.logger.info('开始从DOM提取数据');
      
      const data = {
        bxdzb: {},
        bxmxList: []
      };

      try {
        // 提取报销单号 - 从URL参数提取
        const urlParams = new URLSearchParams(window.location.search);
        const bxdh = urlParams.get('bxdh');
        if (bxdh) {
          data.bxdzb.bxdh = bxdh;
          this.logger.debug('从URL提取报销单号', { bxdh: data.bxdzb.bxdh });
        } else {
          // 尝试多种选择器提取报销单号
          const selectors = [
            '.document-no span:nth-child(3)',
            '.document-no span:last-child',
            '.document-no .form-control',
            '.bxdh-dom .cversionText',
            '[class*="bxdh"] .cversionText'
          ];
          
          for (const selector of selectors) {
            const bxdhEl = document.querySelector(selector);
            if (bxdhEl && bxdhEl.textContent.trim()) {
              data.bxdzb.bxdh = bxdhEl.textContent.trim();
              this.logger.debug('提取报销单号', { bxdh: data.bxdzb.bxdh, selector });
              break;
            }
          }
        }

        // 提取报销人信息 
        const bxrEl = document.querySelector('.bxrxm-dom.text-dom .cversionText');
        if (bxrEl) {
          const text = bxrEl.textContent.trim();
          if (text.includes('/')) {
            const parts = text.split('/');
            data.bxdzb.bxrxm = parts[0].trim();
            data.bxdzb.bxrgh = parts[1].trim();
          } else {
            data.bxdzb.bxrxm = text;
          }
          this.logger.debug('提取报销人信息', { 
            bxrxm: data.bxdzb.bxrxm, 
            bxrgh: data.bxdzb.bxrgh 
          });
        }

        // 如果还没有工号，从收款人信息中提取
        if (!data.bxdzb.bxrgh) {
          // 查找包含收款人信息的元素
          const allElements = document.querySelectorAll('*');
          for (const element of allElements) {
            const text = element.textContent;
            if (text && typeof text === 'string') {
              // 精确匹配：姓名/工号格式
              if (data.bxdzb.bxrxm) {
                const namePattern = data.bxdzb.bxrxm.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
                const match1 = new RegExp(`${namePattern}\/([A-Z0-9]+)`).exec(text);
                if (match1) {
                  data.bxdzb.bxrgh = match1[1];
                  this.logger.debug('精确匹配工号', { 
                    bxrgh: data.bxdzb.bxrgh, 
                    text: text.substring(0, 100) 
                  });
                  break;
                }
              }
              
              // 通用模式：任意姓名/工号（工号长度5-10位）
              const match2 = text.match(/([^\/\s]+)\/([A-Z0-9]{5,10})/);
              if (match2 && match2[1] && match2[2]) {
                const name = match2[1].trim();
                const empNo = match2[2].trim();
                // 验证是否匹配已提取的姓名
                if (data.bxdzb.bxrxm && name === data.bxdzb.bxrxm) {
                  data.bxdzb.bxrgh = empNo;
                  this.logger.debug('通用匹配工号', { 
                    bxrgh: data.bxdzb.bxrgh, 
                    text: text.substring(0, 100) 
                  });
                  break;
                }
              }
            }
          }
        }

        // 提取部门
        const deptEl = document.querySelector('.bxbmmc-dom.text-dom .cversionText');
        if (deptEl) {
          data.bxdzb.bmbmmc = deptEl.textContent.trim();
          this.logger.debug('提取部门', { bmbmmc: data.bxdzb.bmbmmc });
        }

        // 提取公司
        const companyEl = document.querySelector('.fygsgsmc-dom.text-dom .cversionText');
        if (companyEl) {
          data.bxdzb.fygsgsmc = companyEl.textContent.trim();
          this.logger.debug('提取公司', { fygsgsmc: data.bxdzb.fygsgsmc });
        }

        // 提取金额 - 严格从"费用合计"区域提取
        const extractAmount = () => {
          this.logger.debug('💰 开始提取金额...');
          console.log('💰 开始提取金额...');
          
          // 策略1: 严格从包含"费用合计"文字的区域提取（最准确）
          // 第一步：找到包含"费用合计"文字的元素
          let feesSummaryElement = null;
          const allElements = Array.from(document.querySelectorAll('*'));
          
          for (const element of allElements) {
            const text = element.textContent || '';
            // 必须包含"费用合计"这4个字
            if (text.includes('费用合计')) {
              feesSummaryElement = element;
              console.log('💰 ✅ 找到"费用合计"区域');
              break;
            }
          }
          
          if (feesSummaryElement) {
            // 第二步：在"费用合计"区域内提取金额
            const text = feesSummaryElement.textContent || '';
            this.logger.debug(`💰 "费用合计"区域文本: "${text.substring(0, 200)}..."`);
            console.log(`💰 在"费用合计"区域内查找金额...`);
            
            // 匹配模式：应付金额：1,737.80 或 报销总金额：1,737.80
            const amountMatch = text.match(/(?:应付金额|报销总金额|报销总额)[：:\s]*[¥￥]?\s*([\d,]+\.?\d*)/);
            if (amountMatch) {
              const amount = parseFloat(amountMatch[1].replace(/,/g, ''));
              if (!isNaN(amount) && amount > 0) {
                data.bxdzb.bxje = amount;
                this.logger.debug('✅ 从"费用合计"区域提取金额', { 
                  bxje: data.bxdzb.bxje, 
                  matchText: amountMatch[0] 
                });
                console.log(`💰 ✅ 金额提取成功: ${data.bxdzb.bxje}`);
                console.log(`💰 匹配文本: "${amountMatch[0]}"`);
                console.log(`💰 数据来源: "费用合计"区域`);
                return true;
              }
            }
          }
          
          // 策略2: 使用CSS选择器提取（备用方案）
          console.log('💰 "费用合计"区域未找到或无金额，尝试CSS选择器...');
          this.logger.debug('💰 费用合计区域未找到，尝试CSS选择器...');
          const amountSelectors = ['.key-yfjje .right', '.key-grdfje .right', '.bxje .right'];
          for (const selector of amountSelectors) {
            const el = document.querySelector(selector);
            if (el) {
              const text = el.textContent.trim();
              if (text && text !== '0.00' && text !== '0') {
                const amount = parseFloat(text.replace(/[,¥￥\s]/g, ''));
                if (!isNaN(amount) && amount > 0) {
                  data.bxdzb.bxje = amount;
                  this.logger.debug('✅ CSS选择器提取金额', { bxje: data.bxdzb.bxje, selector, text });
                  console.log(`💰 金额提取成功: ${data.bxdzb.bxje} (来源: CSS选择器)`);
                  return true;
                }
              }
            }
          }
          
          this.logger.debug('❌ 未找到金额');
          console.log('💰 未找到金额');
          return false;
        };
        
        extractAmount();

        // 提取发票数量 - 严格从"费用合计"区域提取
        const searchForInvoiceCount = () => {
          this.logger.debug('🎫 开始提取发票数量...');
          console.log('🎫 ================== 开始提取发票数量 ==================');
          
          // 策略1: 严格从包含"费用合计"文字的区域提取（最准确）
          // 第一步：找到包含"费用合计"文字的元素
          console.log('🎫 步骤1: 查找包含"费用合计"文字的元素...');
          
          let feesSummaryElement = null;
          const allElements = Array.from(document.querySelectorAll('*'));
          console.log(`🎫 页面总元素数: ${allElements.length}`);
          
          let foundCount = 0;
          for (const element of allElements) {
            const text = element.textContent || '';
            // 必须包含"费用合计"这4个字
            if (text.includes('费用合计')) {
              foundCount++;
              const textPreview = text.substring(0, 300).replace(/\s+/g, ' ');
              console.log(`🎫 找到第${foundCount}个包含"费用合计"的元素:`);
              console.log(`🎫   - 元素标签: ${element.tagName}`);
              console.log(`🎫   - 元素类名: ${element.className}`);
              console.log(`🎫   - 文本长度: ${text.length}字符`);
              console.log(`🎫   - 文本预览(前300字符): "${textPreview}"`);
              console.log(`🎫   - 是否包含"应付金额": ${text.includes('应付金额')}`);
              console.log(`🎫   - 是否包含"张票据": ${text.includes('张票据')}`);
              
              // 选择一个较小的、同时包含"费用合计"和"应付金额"的元素
              if (text.includes('应付金额') && text.includes('张票据')) {
                // 如果还没有找到，或者当前元素更小（更精确）
                if (!feesSummaryElement || text.length < feesSummaryElement.textContent.length) {
                  feesSummaryElement = element;
                  console.log(`🎫   ✅ 选择此元素作为"费用合计"区域（更精确）`);
                }
              }
            }
          }
          
          if (!feesSummaryElement) {
            console.log('🎫 ❌ 未找到"费用合计"区域');
            this.logger.debug('❌ 未找到"费用合计"区域');
            return false;
          }
          
          console.log('🎫 步骤2: 在选定的"费用合计"区域内提取发票数量');
          console.log(`🎫 最终选定区域文本长度: ${feesSummaryElement.textContent.length}字符`);
          
          // 第二步：在"费用合计"区域内提取发票数量
          const text = feesSummaryElement.textContent || '';
          this.logger.debug(`🎫 "费用合计"区域文本: "${text}"`);
          
          // 打印所有包含"张"的部分
          const allMatches = text.match(/\d+\s*张[^，。]*?/g);
          console.log(`🎫 区域内所有包含"X张"的文本:`, allMatches);
          
          // 匹配模式1：(共12张票据) 或 （共12张票据）
          console.log('🎫 尝试匹配模式1: (共XX张票据)');
          const match1 = text.match(/[（(]共[^张]*?(\d+)[^张]*?张(?:票据|发票|凭证)?[)）]/);
          if (match1) {
            data.bxdzb.fpsl = parseInt(match1[1], 10);
            this.logger.debug('✅ 从"费用合计"区域提取发票数量(模式1)', { 
              fpsl: data.bxdzb.fpsl, 
              matchText: match1[0] 
            });
            console.log(`🎫 ✅ 匹配成功 - 模式1`);
            console.log(`🎫 完整匹配文本: "${match1[0]}"`);
            console.log(`🎫 提取的数字: ${match1[1]}`);
            console.log(`🎫 最终发票数量: ${data.bxdzb.fpsl}张`);
            console.log('🎫 ================== 提取完成 ==================');
            return true;
          }
          console.log('🎫 ❌ 模式1未匹配');
          
          // 匹配模式2：共12张 - 更灵活的匹配
          console.log('🎫 尝试匹配模式2: 共XX张');
          const match2 = text.match(/共[^张]*?(\d+)[^张]*?张(?:票据|发票|凭证)?/);
          if (match2) {
            data.bxdzb.fpsl = parseInt(match2[1], 10);
            this.logger.debug('✅ 从"费用合计"区域提取发票数量(模式2)', { 
              fpsl: data.bxdzb.fpsl, 
              matchText: match2[0] 
            });
            console.log(`🎫 ✅ 匹配成功 - 模式2`);
            console.log(`🎫 完整匹配文本: "${match2[0]}"`);
            console.log(`🎫 提取的数字: ${match2[1]}`);
            console.log(`🎫 最终发票数量: ${data.bxdzb.fpsl}张`);
            console.log('🎫 ================== 提取完成 ==================');
            return true;
          }
          console.log('🎫 ❌ 模式2未匹配');
          
          // 匹配模式3：12张票据
          console.log('🎫 尝试匹配模式3: XX张票据');
          const match3 = text.match(/(\d+)\s*张(?:票据|发票|凭证)/);
          if (match3) {
            data.bxdzb.fpsl = parseInt(match3[1], 10);
            this.logger.debug('✅ 从"费用合计"区域提取发票数量(模式3)', { 
              fpsl: data.bxdzb.fpsl, 
              matchText: match3[0] 
            });
            console.log(`🎫 ✅ 匹配成功 - 模式3`);
            console.log(`🎫 完整匹配文本: "${match3[0]}"`);
            console.log(`🎫 提取的数字: ${match3[1]}`);
            console.log(`🎫 最终发票数量: ${data.bxdzb.fpsl}张`);
            console.log('🎫 ================== 提取完成 ==================');
            return true;
          }
          console.log('🎫 ❌ 模式3未匹配');
          
          console.log('🎫 ❌ 所有模式都未匹配成功');
          console.log('🎫 ================== 提取失败 ==================');
          this.logger.debug('❌ "费用合计"区域内未找到发票数量');
          return false;
        };
        
        if (!searchForInvoiceCount()) {
          // 如果没找到发票数量，尝试统计有效明细条数作为发票数量
          this.logger.debug('未找到发票数量标识，将在明细提取后统计');
        }

        // 提取明细 - 从表格行
        const rows = document.querySelectorAll('table tbody tr');
        this.logger.debug('查找表格行', { rowCount: rows.length });
        
        rows.forEach((row, index) => {
          const cells = row.querySelectorAll('td');
          if (cells.length >= 3) {
            const detail = {};
            
            // 第一列：费用类型/名称
            if (cells[0]) {
              const fymc = cells[0].textContent.trim();
              if (fymc && fymc !== '' && fymc !== '-') {
                detail.fymc = fymc;
              }
            }
            
            // 查找金额列（通常在最后几列）
            for (let i = cells.length - 1; i >= 1; i--) {
              const cellText = cells[i].textContent.trim();
              const amount = parseFloat(cellText.replace(/[,¥￥\s]/g, ''));
              if (!isNaN(amount) && amount > 0) {
                detail.fyje = amount;
                break;
              }
            }
            
            // 查找日期列
            for (let i = 1; i < cells.length - 1; i++) {
              const cellText = cells[i].textContent.trim();
              if (cellText && (cellText.includes('-') || cellText.includes('/')) && cellText.length >= 8) {
                detail.fysj = cellText;
                break;
              }
            }
            
            if (detail.fymc && detail.fyje) {
              data.bxmxList.push(detail);
              this.logger.debug('提取明细', { index, detail });
            }
          }
        });

        // 如果没有从表格提取到明细，尝试从其他可能的结构提取
        if (data.bxmxList.length === 0) {
          this.logger.debug('表格中没有明细，尝试其他结构');
          const detailItems = document.querySelectorAll('.detail-item, .expense-item, .cost-item');
          detailItems.forEach((item, index) => {
            const nameEl = item.querySelector('.name, .title, .desc, .fymc');
            const amountEl = item.querySelector('.amount, .money, .fyje, .je');
            const dateEl = item.querySelector('.date, .time, .fysj');
            
            if (nameEl && amountEl) {
              const detail = {
                fymc: nameEl.textContent.trim(),
                fyje: parseFloat(amountEl.textContent.replace(/[,¥￥\s]/g, '')) || 0,
                fysj: dateEl ? dateEl.textContent.trim() : ''
              };
              
              if (detail.fymc && detail.fyje > 0) {
                data.bxmxList.push(detail);
                this.logger.debug('从其他结构提取明细', { index, detail });
              }
            }
          });
        }

        // 如果还没有发票数量，使用明细条数
        if (!data.bxdzb.fpsl && data.bxmxList.length > 0) {
          data.bxdzb.fpsl = data.bxmxList.length;
          this.logger.debug('使用明细条数作为发票数量', { fpsl: data.bxdzb.fpsl });
        }

        // 检查是否提取到有效数据
        const hasValidData = data.bxdzb.bxdh || data.bxdzb.bxrxm || data.bxmxList.length > 0;
        
        if (hasValidData) {
          // 输出详细的数据汇总信息
          this.logger.info('📊 提取结果汇总:');
          this.logger.info('提取的数据:', data);
          this.logger.info(`数据完整性: ${hasValidData ? '✓ 有效' : '✗ 无效'}`);
          
          this.logger.info('数据统计:');
          this.logger.info(`  - 报销单号: ${data.bxdzb.bxdh || 'N/A'}`);
          this.logger.info(`  - 报销人: ${data.bxdzb.bxrxm || 'N/A'}`);
          this.logger.info(`  - 工号: ${data.bxdzb.bxrgh || 'N/A'}`);
          this.logger.info(`  - 部门: ${data.bxdzb.bmbmmc || 'N/A'}`);
          this.logger.info(`  - 公司: ${data.bxdzb.fygsgsmc || 'N/A'}`);
          this.logger.info(`  - 金额: ${data.bxdzb.bxje || 'N/A'}`);
          this.logger.info(`  - 发票数量: ${data.bxdzb.fpsl || 'N/A'}张`);
          this.logger.info(`  - 明细条数: ${data.bxmxList.length}`);
          
          this.logger.info('✅ DOM数据提取成功', {
            hasBxdh: !!data.bxdzb.bxdh,
            hasBxrxm: !!data.bxdzb.bxrxm,
            hasBxje: !!data.bxdzb.bxje,
            detailCount: data.bxmxList.length
          });
          return data;
        } else {
          this.logger.warn('DOM中未提取到有效数据');
          return null;
        }
      } catch (error) {
        this.logger.error('DOM数据提取失败', { error: error.message });
        return null;
      }
    }

    /**
     * 提取基础信息
     */
    async extractBasicInfo(pageInfo, jsData = null) {
      this.logger.info('提取基础信息');

      // 如果有JS数据，优先使用
      if (jsData?.bxdzb) {
        const bxdzb = jsData.bxdzb;
        return {
          company: bxdzb.fygsgsmc || bxdzb.fygsmc || '公司名称',
          bxdh: pageInfo.bxdh,
          type: pageInfo.type,
          date: bxdzb.bxrq || bxdzb.fssj || new Date().toISOString().split('T')[0],
        };
      }

      // 否则从DOM提取
      const company = await this.extractField('company', SELECTORS.company);

      return {
        company: company || '公司名称',
        bxdh: pageInfo.bxdh,
        type: pageInfo.type,
        date: new Date().toISOString().split('T')[0],
      };
    }

    /**
     * 提取员工信息
     */
    async extractEmployeeInfo(jsData = null) {
      this.logger.info('提取员工信息');

      // 如果有JS数据，优先使用
      if (jsData?.bxdzb) {
        const bxdzb = jsData.bxdzb;
        // 解析工号（格式：姓名/工号）
        let id = bxdzb.bxrgh || '';
        let name = bxdzb.bxrxm || '';

        // 处理"姓名/工号"格式
        if (bxdzb.bxrgh && bxdzb.bxrgh.includes('/')) {
          const parts = bxdzb.bxrgh.split('/');
          name = parts[0] || name;
          id = parts[1] || bxdzb.bxrgh;
        }

        return {
          id: id || '',
          name: name || '',
          department: bxdzb.tdbmmc || bxdzb.bmbm || bxdzb.bxbm || '',
          departmentCode: bxdzb.tdbumid || bxdzb.bmbh || '',
        };
      }

      // 否则从DOM提取
      const id = await this.extractField('employeeId', SELECTORS.employeeId);
      const name = await this.extractField('employeeName', SELECTORS.employeeName);
      const department = await this.extractField('department', SELECTORS.department);
      const departmentCode = await this.extractField('departmentCode', SELECTORS.departmentCode);

      return {
        id: id || '',
        name: name || '',
        department: department || '',
        departmentCode: departmentCode || '',
      };
    }

    /**
     * 提取费用明细 - 改进版本
     */
    async extractDetails(jsData = null) {
      this.logger.info('提取费用明细');

      // 如果有JS数据，优先使用API数据
      if (jsData?.bxmxList && Array.isArray(jsData.bxmxList)) {
        const details = jsData.bxmxList.map((item, index) => ({
          序号: index + 1,
          用途说明: item.fymc || item.fylxmc || '',
          发生日期: this.formatDate(item.xfsj || item.fssj),
          金额: parseAmount(item.xfje || item.fyje || item.bxje),
          备注: item.bz || item.fysm || item.xcxx || '',
        }));

        this.logger.info(`从JS数据成功提取 ${details.length} 条明细`);
        return details;
      }

      // 从DOM提取
      const details = this.extractDetailsFromDOM();
      return details;
    }

    /**
     * 从DOM提取明细数据
     */
    extractDetailsFromDOM() {
      const details = [];

      // 查找Element UI表格
      const tables = document.querySelectorAll('.el-table__body-wrapper table, .el-table__body table');
      if (tables.length === 0) {
        this.logger.warn('未找到表格');
        return details;
      }

      for (const table of tables) {
        const rows = table.querySelectorAll('tr');
        for (let i = 0; i < rows.length; i++) {
          const row = rows[i];
          // 跳过表头行
          if (row.closest('thead')) continue;
          if (row.querySelector('th')) continue;

          const cells = row.querySelectorAll('td');
          if (cells.length === 0) continue;

          const detail = this.parseDetailRow(cells, details.length + 1);
          if (detail.用途说明 || detail.金额 > 0) {
            details.push(detail);
          }
        }
      }

      // 如果上面的方法没找到，尝试其他方式
      if (details.length === 0) {
        // 尝试查找所有.el-table__row
        const elTableRows = document.querySelectorAll('.el-table__row');
        for (let i = 0; i < elTableRows.length; i++) {
          const row = elTableRows[i];
          // 获取所有单元格
          const cells = row.querySelectorAll('td');
          if (cells.length > 0) {
            const detail = this.parseDetailRow(cells, details.length + 1);
            if (detail.用途说明 || detail.金额 > 0) {
              details.push(detail);
            }
          }
        }
      }

      // 仍然没找到，尝试从展开的明细行获取
      if (details.length === 0) {
        const expandedRows = document.querySelectorAll('.el-table__expanded-cell');
        for (const expanded of expandedRows) {
          const text = expanded.textContent;
          // 尝试解析展开行中的数据
          if (text) {
            // 查找金额数字
            const amountMatch = text.match(/(\d+\.?\d*)\s*元/);
            if (amountMatch) {
              details.push({
                序号: details.length + 1,
                用途说明: '展开明细',
                发生日期: '',
                金额: parseAmount(amountMatch[1]),
                备注: text.substring(0, 50),
              });
            }
          }
        }
      }

      this.logger.info(`从DOM成功提取 ${details.length} 条明细`);
      return details;
    }

    /**
     * 格式化日期
     */
    formatDate(dateStr) {
      if (!dateStr) return '';
      // 处理 "2025-12-28 00:00:00" 格式
      const match = dateStr.match(/(\d{4}-\d{2}-\d{2})/);
      return match ? match[1] : dateStr;
    }

    /**
     * 解析明细行 - 支持Element UI
     */
    parseDetailRow(cells, index) {
      const detail = {
        序号: index,
        用途说明: '',
        发生日期: '',
        金额: 0,
        备注: '',
      };

      // 尝试多种方式获取单元格文本
      const getCellText = (cell) => {
        // 先尝试 .cell 容器
        const cellElement = cell.querySelector('.cell');
        if (cellElement) {
          return cellElement.textContent.trim();
        }
        return cell.textContent.trim();
      };

      // 根据列数动态解析
      if (cells.length >= 2) {
        detail.用途说明 = getCellText(cells[0]);
        detail.金额 = parseAmount(getCellText(cells[cells.length - 1]));
      }

      if (cells.length >= 3) {
        detail.发生日期 = getCellText(cells[1]);
      }

      if (cells.length >= 4) {
        detail.备注 = getCellText(cells[cells.length - 2]);
      }

      return detail;
    }

    /**
     * 提取汇总信息
     */
    async extractSummary(jsData = null) {
      this.logger.info('提取汇总信息');

      // 如果有JS数据，优先使用
      if (jsData?.bxdzb) {
        const bxdzb = jsData.bxdzb;
        const actualAmount = parseAmount(bxdzb.bxje || bxdzb.spbxje || 0);
        const totalAmount = parseAmount(bxdzb.xfzje || bxdzb.grdfje || actualAmount);
        const voucherCount = parseInt(bxdzb.bxpzzs || bxdzb.pzzs || 0);

        return {
          voucherCount,
          totalAmount,
          actualAmount,
          paidAmount: 0,
          unpaidAmount: actualAmount,
        };
      }

      // 否则从DOM提取
      const actualAmountText = await this.extractField('actualAmount', SELECTORS.actualAmount);
      const totalAmountText = await this.extractField('totalAmount', SELECTORS.totalAmount);
      const voucherCountText = await this.extractField('voucherCount', SELECTORS.voucherCount);

      const actualAmount = parseAmount(actualAmountText);
      const totalAmount = parseAmount(totalAmountText) || actualAmount;
      const voucherCount = parseInt(voucherCountText) || 0;

      return {
        voucherCount,
        totalAmount,
        actualAmount,
        paidAmount: 0,
        unpaidAmount: actualAmount,
      };
    }

    /**
     * 提取支付信息
     */
    async extractPayment() {
      this.logger.info('提取支付信息');

      return {
        method: '电汇',
        accountNo: '',
      };
    }

    /**
     * 提取单个字段（支持重试）
     */
    async extractField(fieldName, selectors, maxRetries = PERFORMANCE_CONFIG.maxRetries) {
      // 检查缓存
      if (this.selectorCache.has(fieldName)) {
        const cachedSelector = this.selectorCache.get(fieldName);
        const value = this.tryExtract(cachedSelector);
        if (value) return value;
      }

      // 遍历选择器
      for (const selector of selectors) {
        const value = await this.waitAndExtract(selector, 3000);
        if (value) {
          this.selectorCache.set(fieldName, selector);
          this.logger.debug(`字段 ${fieldName} 使用选择器: ${selector}`);
          return value;
        }
      }

      // 重试
      if (maxRetries > 0) {
        this.logger.warn(`字段 ${fieldName} 提取失败，重试中...`);
        await sleep(PERFORMANCE_CONFIG.retryDelay);
        return this.extractField(fieldName, selectors, maxRetries - 1);
      }

      this.logger.warn(`无法提取字段: ${fieldName}`);
      return null;
    }

    /**
     * 尝试提取（不等待）
     */
    tryExtract(selector) {
      try {
        const element = document.querySelector(selector);
        if (element) {
          return getElementText(element);
        }
      } catch (error) {
        // 忽略错误
      }
      return null;
    }

    /**
     * 等待并提取
     */
    async waitAndExtract(selector, timeout) {
      return new Promise((resolve) => {
        const element = document.querySelector(selector);
        if (element) {
          resolve(getElementText(element));
          return;
        }

        const observer = new MutationObserver(() => {
          const element = document.querySelector(selector);
          if (element) {
            observer.disconnect();
            resolve(getElementText(element));
          }
        });

        observer.observe(document.body, {
          childList: true,
          subtree: true,
        });

        setTimeout(() => {
          observer.disconnect();
          resolve(null);
        }, timeout);
      });
    }

    /**
     * 查找元素
     */
    async findElement(selectors, timeout = 5000) {
      for (const selector of selectors) {
        const element = document.querySelector(selector);
        if (element) {
          return element;
        }
      }

      // 等待元素出现
      return new Promise((resolve) => {
        const observer = new MutationObserver(() => {
          for (const selector of selectors) {
            const element = document.querySelector(selector);
            if (element) {
              observer.disconnect();
              resolve(element);
              return;
            }
          }
        });

        observer.observe(document.body, {
          childList: true,
          subtree: true,
        });

        setTimeout(() => {
          observer.disconnect();
          resolve(null);
        }, timeout);
      });
    }

    /**
     * 🆕 提取费用类别
     * 从页面标题、DOM元素、URL等多个源提取费用类别信息
     */
    async extractFeeCategory(jsData = null) {
      console.log('💰 开始提取费用类别...');
      
      // 1. 优先从JS数据中提取
      if (jsData?.bxdzb?.fylb) {
        console.log('✅ 从JS数据中找到费用类别:', jsData.bxdzb.fylb);
        return jsData.bxdzb.fylb;
      }

      // 2. 费用类别关键词匹配
      const feeTypeKeywords = [
        '日常费用', '差旅费用', '招待费用', '培训费用', 
        '办公费用', '通讯费用', '交通费用', '住宿费用',
        '餐饮费用', '会议费用', '其他费用'
      ];

      // 3. 从页面标题提取
      const pageTitle = document.title;
      console.log('📄 页面标题:', pageTitle);
      for (const keyword of feeTypeKeywords) {
        if (pageTitle.includes(keyword)) {
          console.log('✅ 从页面标题找到费用类别:', keyword);
          return keyword;
        }
      }

      // 4. 从所有DOM元素中搜索
      console.log('🔍 从DOM元素中搜索费用类别...');
      const allElements = document.querySelectorAll('*');
      for (const element of allElements) {
        const text = element.textContent;
        if (text && typeof text === 'string') {
          for (const keyword of feeTypeKeywords) {
            if (text.includes(keyword)) {
              console.log('✅ 从DOM元素找到费用类别:', keyword, '(元素文本:', text.trim().substring(0, 100), ')');
              return keyword;
            }
          }
        }
      }

      // 5. 从URL推断
      const pageUrl = window.location.href.toLowerCase();
      console.log('🔗 页面URL:', pageUrl);
      if (pageUrl.includes('travel') || pageUrl.includes('差旅')) {
        console.log('✅ 从URL推断费用类别: 差旅费用');
        return '差旅费用';
      }

      // 6. 默认值
      console.log('⚠️ 未找到费用类别，使用默认值: 日常费用');
      return '日常费用';
    }

    /**
     * 🆕 构建bxdzb（报销单主表）数据结构
     */
    buildBxdzb(data, jsData, feeCategory) {
      console.log('🏗️ 开始构建bxdzb数据结构...');
      
      // 从URL提取报销单号（优先）
      const urlParams = new URLSearchParams(window.location.search);
      const bxdhFromUrl = urlParams.get('bxdh');
      
      const bxdzb = {
        // 报销单号
        bxdh: bxdhFromUrl || data.basic?.bxdh || jsData?.bxdzb?.bxdh || 'N/A',
        
        // 报销人信息
        bxrxm: data.employee?.name || jsData?.bxdzb?.bxrxm || 'N/A',
        bxrgh: data.employee?.empNo || jsData?.bxdzb?.bxrgh || 'N/A',
        
        // 部门和公司
        bmbmmc: data.employee?.department || jsData?.bxdzb?.bmbmmc || 'N/A',
        fygsgsmc: data.basic?.company || jsData?.bxdzb?.fygsgsmc || 'N/A',
        
        // 金额信息
        bxje: data.summary?.totalAmount || jsData?.bxdzb?.bxje || 0,
        
        // 发票数量
        fpsl: jsData?.bxdzb?.fpsl || data.details?.length || 0,
        
        // 🆕 费用类别
        fylb: feeCategory || jsData?.bxdzb?.fylb || '日常费用'
      };

      console.log('✅ bxdzb构建完成:', bxdzb);
      return bxdzb;
    }

    /**
     * 🆕 构建bxmxList（报销明细列表）数据结构
     */
    buildBxmxList(data, jsData, feeCategory) {
      console.log('🏗️ 开始构建bxmxList数据结构...');
      
      // 优先使用JS数据中的明细
      if (jsData?.bxmxList && Array.isArray(jsData.bxmxList)) {
        console.log('✅ 使用JS数据中的bxmxList:', jsData.bxmxList.length, '条');
        return jsData.bxmxList.map(item => ({
          ...item,
          feeType: feeCategory || item.feeType || '日常费用'
        }));
      }

      // 使用提取的details数据构建
      if (data.details && Array.isArray(data.details)) {
        console.log('✅ 从details构建bxmxList:', data.details.length, '条');
        return data.details.map((item, index) => ({
          fymc: item.description || item.fymc || `费用${index + 1}`,
          fyje: item.amount || item.fyje || 0,
          fysj: item.date || item.fysj || new Date().toLocaleDateString('zh-CN'),
          feeType: feeCategory || item.feeType || '日常费用',
          description: item.description || item.fymc,
          amount: item.amount || item.fyje
        }));
      }

      console.log('⚠️ 无明细数据，返回空数组');
      return [];
    }
  }

  /**
   * UI注入器 - 在页面中注入交互元素
   */


  class UIInjector {
    constructor() {
      this.logger = new Logger('UIInjector');
      this.floatingButton = null;
      this.toast = null;
    }

    /**
     * 注入悬浮按钮
     */
    injectFloatingButton(onClick) {
      this.logger.info('注入悬浮按钮');

      // 避免重复注入
      if (this.floatingButton) {
        return this.floatingButton;
      }

      const button = document.createElement('button');
      button.id = 'reimbursement-assistant-button';
      button.textContent = '生成报销底单';
      button.style.cssText = `
      position: fixed;
      bottom: 20px;
      right: 20px;
      z-index: 10000;
      padding: 12px 24px;
      background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
      color: white;
      border: none;
      border-radius: 8px;
      font-size: 14px;
      font-weight: 600;
      cursor: pointer;
      box-shadow: 0 4px 12px rgba(102, 126, 234, 0.4);
      transition: all 0.3s ease;
      font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
    `;

      // 添加悬停效果
      button.addEventListener('mouseenter', () => {
        button.style.transform = 'translateY(-2px)';
        button.style.boxShadow = '0 6px 16px rgba(102, 126, 234, 0.5)';
      });

      button.addEventListener('mouseleave', () => {
        button.style.transform = 'translateY(0)';
        button.style.boxShadow = '0 4px 12px rgba(102, 126, 234, 0.4)';
      });

      // 点击事件
      button.addEventListener('click', () => {
        this.logger.info('悬浮按钮被点击');
        if (onClick) onClick();
      });

      document.body.appendChild(button);
      this.floatingButton = button;

      return button;
    }

    /**
     * 更新按钮状态
     */
    updateButtonStatus(status) {
      if (!this.floatingButton) return;

      const statusConfig = {
        [PROCESS_STATUS.IDLE]: {
          text: '生成报销底单',
          color: 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)',
          disabled: false,
        },
        [PROCESS_STATUS.DETECTING]: {
          text: '检测中...',
          color: 'linear-gradient(135deg, #f093fb 0%, #f5576c 100%)',
          disabled: true,
        },
        [PROCESS_STATUS.EXTRACTING]: {
          text: '提取数据中...',
          color: 'linear-gradient(135deg, #4facfe 0%, #00f2fe 100%)',
          disabled: true,
        },
        [PROCESS_STATUS.GENERATING]: {
          text: '生成PDF中...',
          color: 'linear-gradient(135deg, #43e97b 0%, #38f9d7 100%)',
          disabled: true,
        },
        [PROCESS_STATUS.COMPLETED]: {
          text: '✓ 生成成功',
          color: 'linear-gradient(135deg, #11998e 0%, #38ef7d 100%)',
          disabled: false,
        },
        [PROCESS_STATUS.ERROR]: {
          text: '✗ 生成失败',
          color: 'linear-gradient(135deg, #eb3349 0%, #f45c43 100%)',
          disabled: false,
        },
      };

      const config = statusConfig[status] || statusConfig[PROCESS_STATUS.IDLE];

      this.floatingButton.textContent = config.text;
      this.floatingButton.style.background = config.color;
      this.floatingButton.disabled = config.disabled;
      this.floatingButton.style.cursor = config.disabled ? 'not-allowed' : 'pointer';
      this.floatingButton.style.opacity = config.disabled ? '0.8' : '1';

      this.logger.info('按钮状态已更新', { status, config });
    }

    /**
     * 显示提示消息
     */
    showToast(message, type = 'info', duration = 3000) {
      this.logger.info('显示提示消息', { message, type });

      // 移除旧的toast
      if (this.toast) {
        this.toast.remove();
      }

      const toast = document.createElement('div');
      toast.textContent = message;

      const colors = {
        info: '#1890ff',
        success: '#52c41a',
        warning: '#faad14',
        error: '#f5222d',
      };

      toast.style.cssText = `
      position: fixed;
      top: 20px;
      left: 50%;
      transform: translateX(-50%);
      z-index: 10001;
      padding: 12px 24px;
      background: white;
      color: ${colors[type] || colors.info};
      border-left: 4px solid ${colors[type] || colors.info};
      border-radius: 4px;
      font-size: 14px;
      box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
      animation: slideDown 0.3s ease;
      font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
    `;

      // 添加动画
      const style = document.createElement('style');
      style.textContent = `
      @keyframes slideDown {
        from {
          opacity: 0;
          transform: translateX(-50%) translateY(-20px);
        }
        to {
          opacity: 1;
          transform: translateX(-50%) translateY(0);
        }
      }
      @keyframes slideUp {
        from {
          opacity: 1;
          transform: translateX(-50%) translateY(0);
        }
        to {
          opacity: 0;
          transform: translateX(-50%) translateY(-20px);
        }
      }
    `;
      document.head.appendChild(style);

      document.body.appendChild(toast);
      this.toast = toast;

      // 自动移除
      if (duration > 0) {
        setTimeout(() => {
          toast.style.animation = 'slideUp 0.3s ease';
          setTimeout(() => {
            toast.remove();
            if (this.toast === toast) {
              this.toast = null;
            }
          }, 300);
        }, duration);
      }
    }

    /**
     * 显示进度指示器
     */
    showProgress(message) {
      this.logger.info('显示进度指示器', { message });

      const progress = document.createElement('div');
      progress.id = 'reimbursement-assistant-progress';
      progress.innerHTML = `
      <div style="text-align: center;">
        <div class="spinner"></div>
        <div style="margin-top: 12px; font-size: 14px; color: #666;">${message}</div>
      </div>
    `;

      progress.style.cssText = `
      position: fixed;
      top: 50%;
      left: 50%;
      transform: translate(-50%, -50%);
      z-index: 10002;
      padding: 32px 48px;
      background: white;
      border-radius: 8px;
      box-shadow: 0 8px 24px rgba(0, 0, 0, 0.15);
      font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
    `;

      // 添加spinner样式
      const style = document.createElement('style');
      style.textContent = `
      .spinner {
        border: 3px solid #f3f3f3;
        border-top: 3px solid #667eea;
        border-radius: 50%;
        width: 40px;
        height: 40px;
        animation: spin 1s linear infinite;
        margin: 0 auto;
      }
      @keyframes spin {
        0% { transform: rotate(0deg); }
        100% { transform: rotate(360deg); }
      }
    `;
      document.head.appendChild(style);

      document.body.appendChild(progress);
      return progress;
    }

    /**
     * 隐藏进度指示器
     */
    hideProgress() {
      const progress = document.getElementById('reimbursement-assistant-progress');
      if (progress) {
        progress.remove();
        this.logger.info('隐藏进度指示器');
      }
    }

    /**
     * 移除所有注入的UI
     */
    cleanup() {
      this.logger.info('清理注入的UI元素');

      if (this.floatingButton) {
        this.floatingButton.remove();
        this.floatingButton = null;
      }

      if (this.toast) {
        this.toast.remove();
        this.toast = null;
      }

      this.hideProgress();
    }
  }

  /**
   * Content Script主入口
   */


  class ContentScript {
    constructor() {
      this.logger = new Logger('ContentScript');
      this.detector = new PageDetector();
      this.extractor = new DataExtractor();
      this.uiInjector = new UIInjector();
      this.pageInfo = null;
      this.isProcessing = false;
    }

    /**
     * 初始化
     */
    async init() {
      this.logger.info('Content Script 初始化');

      try {
        // 启动网络拦截器（需要在页面API调用之前启动）
        this.extractor.initNetworkInterceptor();

        // 添加消息监听器用于测试
        this.setupMessageListener();

        // 等待页面加载
        await this.detector.waitForPageReady();

        // 等待报销单数据加载完成（因为数据通过API动态加载）
        await this.detector.waitForDataReady();

        // 检测页面
        this.pageInfo = this.detector.detectPage();

        if (!this.pageInfo) {
          this.logger.warn('当前页面不是报销单页面');
          return;
        }

        this.logger.info('报销单页面已识别', this.pageInfo);

        // 通知Background
        chrome.runtime.sendMessage({
          type: MESSAGE_TYPES.PAGE_DETECTED,
          data: this.pageInfo,
        });

        // 注入UI
        this.setupUI();

        // 监听来自Background的消息
        this.setupMessageListener();
      } catch (error) {
        this.logger.error('初始化失败', { error: error.message });
      }
    }

    /**
     * 设置UI
     */
    setupUI() {
      // 注入悬浮按钮
      this.uiInjector.injectFloatingButton(() => {
        this.handleGenerate();
      });

      // 显示提示
      this.uiInjector.showToast('报销助手已就绪，点击右下角按钮生成PDF', 'success');
    }

    /**
     * 设置消息监听器
     */
    setupMessageListener() {
      chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
        this.logger.info('收到消息', { type: message.type });

        switch (message.type) {
          case MESSAGE_TYPES.START_EXTRACT:
            this.handleExtract().then(sendResponse);
            return true; // 异步响应

          default:
            sendResponse({ success: false, error: 'Unknown message type' });
        }
      });

      // 监听来自页面的消息（用于测试）
      window.addEventListener('message', async (event) => {
        if (event.source !== window) return;
        
        if (event.data.type === 'TEST_TEMPLATE_ACCESS') {
          this.logger.info('收到模板访问测试请求');
          try {
            // 测试模板文件访问
            const templatePath = chrome.runtime.getURL('dt-resources/baoXiaoDanModel.pdf');
            const response = await fetch(templatePath);
            
            const result = {
              type: 'TEST_TEMPLATE_RESULT',
              success: response.ok,
              status: response.status,
              statusText: response.statusText,
              templatePath: templatePath
            };
            
            if (response.ok) {
              const buffer = await response.arrayBuffer();
              result.size = buffer.byteLength;
              
              // 验证PDF格式
              const uint8Array = new Uint8Array(buffer);
              result.isPDF = uint8Array[0] === 0x25 && uint8Array[1] === 0x50 && 
                            uint8Array[2] === 0x44 && uint8Array[3] === 0x46;
            }
            
            // 发送结果回页面
            window.postMessage(result, '*');
            this.logger.info('模板访问测试完成', result);
          } catch (error) {
            window.postMessage({
              type: 'TEST_TEMPLATE_RESULT',
              success: false,
              error: error.message
            }, '*');
          }
        }
        
        if (event.data.type === 'GENERATE_PDF_TEST') {
          this.logger.info('收到PDF生成测试请求');
          try {
            // 使用测试数据生成PDF
            const testData = event.data.data;
            
            // 发送到Background生成PDF
            const response = await chrome.runtime.sendMessage({
              type: MESSAGE_TYPES.DATA_EXTRACTED,
              data: testData,
            });
            
            const result = {
              type: 'PDF_GENERATION_RESULT',
              success: response.success
            };
            
            if (response.success) {
              result.size = response.size || '未知';
              result.variableCount = Object.keys(testData.bxdzb || {}).length;
              result.method = response.method || '模板生成';
            } else {
              result.error = response.error || 'PDF生成失败';
            }
            
            // 发送结果回页面
            window.postMessage(result, '*');
            this.logger.info('PDF生成测试完成', result);
          } catch (error) {
            window.postMessage({
              type: 'PDF_GENERATION_RESULT',
              success: false,
              error: error.message
            }, '*');
          }
        }
      });
    }

    /**
     * 处理生成PDF请求
     */
    async handleGenerate() {
      if (this.isProcessing) {
        this.uiInjector.showToast('正在处理中，请稍候...', 'warning');
        return;
      }

      this.isProcessing = true;
      this.uiInjector.updateButtonStatus(PROCESS_STATUS.EXTRACTING);

      try {
        // 显示进度
        const progress = this.uiInjector.showProgress('正在提取数据...');

        // 提取数据
        const data = await this.extractor.extract(this.pageInfo);

        // 更新进度
        progress.querySelector('div:last-child').textContent = '正在生成PDF...';
        this.uiInjector.updateButtonStatus(PROCESS_STATUS.GENERATING);

        // 🚨 强制调试：发送数据到Background
        console.log('%c📤 Content Script正在发送数据到Background!', 'color: #f39c12; font-size: 14px; font-weight: bold;');
        console.log('消息类型:', MESSAGE_TYPES.DATA_EXTRACTED);
        console.log('数据摘要:', {
          bxdh: data.bxdzb?.bxdh,
          bxrxm: data.bxdzb?.bxrxm,
          fylb: data.bxdzb?.fylb,
          明细条数: data.bxmxList?.length
        });
        
        // 发送数据到Background生成PDF
        const response = await chrome.runtime.sendMessage({
          type: MESSAGE_TYPES.DATA_EXTRACTED,
          data: data,
        });

        console.log('%c📨 Background响应:', 'color: #27ae60; font-size: 14px; font-weight: bold;', response);

        this.uiInjector.hideProgress();

        if (response.success) {
          this.uiInjector.updateButtonStatus(PROCESS_STATUS.COMPLETED);
          this.uiInjector.showToast('PDF生成成功！', 'success');

          // 3秒后恢复按钮状态
          setTimeout(() => {
            this.uiInjector.updateButtonStatus(PROCESS_STATUS.IDLE);
          }, 3000);
        } else {
          throw new Error(response.error || 'PDF生成失败');
        }
      } catch (error) {
        this.logger.error('生成失败', { error: error.message });
        this.uiInjector.hideProgress();
        this.uiInjector.updateButtonStatus(PROCESS_STATUS.ERROR);
        this.uiInjector.showToast(`生成失败: ${error.message}`, 'error', 5000);

        // 5秒后恢复按钮状态
        setTimeout(() => {
          this.uiInjector.updateButtonStatus(PROCESS_STATUS.IDLE);
        }, 5000);
      } finally {
        this.isProcessing = false;
      }
    }

    /**
     * 处理提取数据请求
     */
    async handleExtract() {
      try {
        const data = await this.extractor.extract(this.pageInfo);
        return { success: true, data };
      } catch (error) {
        return { success: false, error: error.message };
      }
    }
  }

  // 启动Content Script
  const contentScript = new ContentScript();
  contentScript.init();

})();
